/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.controller.arphandler.internal;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.LinkedBlockingQueue;
import org.opendaylight.controller.arphandler.ARPCacheEvent;
import org.opendaylight.controller.arphandler.ARPEvent;
import org.opendaylight.controller.arphandler.ARPReply;
import org.opendaylight.controller.arphandler.ARPRequest;
import org.opendaylight.controller.clustering.services.CacheConfigException;
import org.opendaylight.controller.clustering.services.CacheExistException;
import org.opendaylight.controller.clustering.services.ICacheUpdateAware;
import org.opendaylight.controller.clustering.services.IClusterContainerServices;
import org.opendaylight.controller.clustering.services.IClusterServices;
import org.opendaylight.controller.connectionmanager.IConnectionManager;
import org.opendaylight.controller.hosttracker.HostIdFactory;
import org.opendaylight.controller.hosttracker.IHostId;
import org.opendaylight.controller.hosttracker.IfHostListener;
import org.opendaylight.controller.hosttracker.IfIptoHost;
import org.opendaylight.controller.hosttracker.hostAware.HostNodeConnector;
import org.opendaylight.controller.hosttracker.hostAware.IHostFinder;
import org.opendaylight.controller.sal.connection.ConnectionLocality;
import org.opendaylight.controller.sal.core.ConstructionException;
import org.opendaylight.controller.sal.core.Node;
import org.opendaylight.controller.sal.core.NodeConnector;
import org.opendaylight.controller.sal.packet.ARP;
import org.opendaylight.controller.sal.packet.Ethernet;
import org.opendaylight.controller.sal.packet.IDataPacketService;
import org.opendaylight.controller.sal.packet.IListenDataPacket;
import org.opendaylight.controller.sal.packet.IPv4;
import org.opendaylight.controller.sal.packet.Packet;
import org.opendaylight.controller.sal.packet.PacketResult;
import org.opendaylight.controller.sal.packet.RawPacket;
import org.opendaylight.controller.sal.routing.IRouting;
import org.opendaylight.controller.sal.utils.EtherTypes;
import org.opendaylight.controller.sal.utils.HexEncode;
import org.opendaylight.controller.sal.utils.NetUtils;
import org.opendaylight.controller.switchmanager.ISwitchManager;
import org.opendaylight.controller.switchmanager.Subnet;
import org.opendaylight.controller.topologymanager.ITopologyManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ArpHandler
implements IHostFinder,
IListenDataPacket,
ICacheUpdateAware<ARPEvent, Boolean> {
    private static final Logger log = LoggerFactory.getLogger(ArpHandler.class);
    static final String ARP_EVENT_CACHE_NAME = "arphandler.arpRequestReplyEvent";
    private IfIptoHost hostTracker;
    private ISwitchManager switchManager;
    private ITopologyManager topologyManager;
    private IDataPacketService dataPacketService;
    private IRouting routing;
    private IClusterContainerServices clusterContainerService;
    private IConnectionManager connectionManager;
    private Set<IfHostListener> hostListeners = new CopyOnWriteArraySet<IfHostListener>();
    private ConcurrentMap<InetAddress, Set<HostNodeConnector>> arpRequestors;
    private ConcurrentMap<InetAddress, Short> countDownTimers;
    private Timer periodicTimer;
    private BlockingQueue<ARPCacheEvent> ARPCacheEvents = new LinkedBlockingQueue<ARPCacheEvent>();
    private Thread cacheEventHandler;
    private boolean stopping = false;
    private ConcurrentMap<ARPEvent, Boolean> arpRequestReplyEvent;

    void setConnectionManager(IConnectionManager cm) {
        this.connectionManager = cm;
    }

    void unsetConnectionManager(IConnectionManager cm) {
        if (this.connectionManager == cm) {
            this.connectionManager = null;
        }
    }

    void setClusterContainerService(IClusterContainerServices s) {
        this.clusterContainerService = s;
    }

    void unsetClusterContainerService(IClusterContainerServices s) {
        if (this.clusterContainerService == s) {
            this.clusterContainerService = null;
        }
    }

    void setRouting(IRouting r) {
        this.routing = r;
    }

    void unsetRouting(IRouting r) {
        if (this.routing == r) {
            this.routing = null;
        }
    }

    void setHostListener(IfHostListener s) {
        if (this.hostListeners != null) {
            this.hostListeners.add(s);
        }
    }

    void unsetHostListener(IfHostListener s) {
        if (this.hostListeners != null) {
            this.hostListeners.remove(s);
        }
    }

    void setDataPacketService(IDataPacketService s) {
        this.dataPacketService = s;
    }

    void unsetDataPacketService(IDataPacketService s) {
        if (this.dataPacketService == s) {
            this.dataPacketService = null;
        }
    }

    public void setHostTracker(IfIptoHost hostTracker) {
        log.debug("Setting HostTracker");
        this.hostTracker = hostTracker;
    }

    public void unsetHostTracker(IfIptoHost s) {
        log.debug("UNSetting HostTracker");
        if (this.hostTracker == s) {
            this.hostTracker = null;
        }
    }

    public void setTopologyManager(ITopologyManager tm) {
        this.topologyManager = tm;
    }

    public void unsetTopologyManager(ITopologyManager tm) {
        if (this.topologyManager == tm) {
            this.topologyManager = null;
        }
    }

    protected void sendARPReply(NodeConnector p, byte[] sMAC, InetAddress sIP, byte[] tMAC, InetAddress tIP) {
        byte[] senderIP = sIP.getAddress();
        byte[] targetIP = tIP.getAddress();
        ARP arp = this.createARP(ARP.REPLY, sMAC, senderIP, tMAC, targetIP);
        Ethernet ethernet = this.createEthernet(sMAC, tMAC, arp);
        RawPacket destPkt = this.dataPacketService.encodeDataPacket((Packet)ethernet);
        destPkt.setOutgoingNodeConnector(p);
        this.dataPacketService.transmitDataPacket(destPkt);
    }

    protected void handleARPPacket(Ethernet eHeader, ARP pkt, NodeConnector p) {
        InetAddress sourceIP;
        InetAddress targetIP;
        byte[] sourceMAC = eHeader.getSourceMACAddress();
        byte[] targetMAC = eHeader.getDestinationMACAddress();
        if (Arrays.equals(sourceMAC, this.getControllerMAC())) {
            if (log.isDebugEnabled()) {
                log.debug("Receive a self originated ARP pkt (srcMAC {}) --> DROP", (Object)HexEncode.bytesToHexString((byte[])sourceMAC));
            }
            return;
        }
        try {
            targetIP = InetAddress.getByAddress(pkt.getTargetProtocolAddress());
            sourceIP = InetAddress.getByAddress(pkt.getSenderProtocolAddress());
        }
        catch (UnknownHostException e1) {
            log.debug("Invalid host in ARP packet: {}", (Object)e1.getMessage());
            return;
        }
        Subnet subnet = null;
        if (this.switchManager != null) {
            subnet = this.switchManager.getSubnetByNetworkAddress(sourceIP);
        }
        if (subnet == null) {
            log.debug("ARPHandler: can't find subnet matching {}, drop packet", (Object)sourceIP);
            return;
        }
        if (!subnet.hasNodeConnector(p)) {
            log.debug("{} showing up on {} does not belong to {}", new Object[]{sourceIP, p, subnet});
            return;
        }
        HostNodeConnector requestor = null;
        if (NetUtils.isUnicastMACAddr((byte[])sourceMAC) && p.getNode() != null) {
            try {
                requestor = new HostNodeConnector(sourceMAC, sourceIP, p, subnet.getVlan());
            }
            catch (ConstructionException e) {
                log.debug("Received ARP packet with invalid MAC: {}", (Object)HexEncode.bytesToHexString((byte[])sourceMAC));
                return;
            }
            log.trace("Inform Host tracker of new host {}", (Object)requestor.getNetworkAddress());
            for (IfHostListener listener : this.hostListeners) {
                listener.hostListener(requestor);
            }
        }
        if (pkt.getOpCode() != ARP.REQUEST || sourceIP.equals(targetIP)) {
            log.trace("Received ARP reply packet from {}, reply to all requestors.", (Object)sourceIP);
            this.arpRequestReplyEvent.put(new ARPReply(sourceIP, sourceMAC), true);
            return;
        }
        if (targetIP.equals(subnet.getNetworkAddress()) && (NetUtils.isBroadcastMACAddr((byte[])targetMAC) || Arrays.equals(targetMAC, this.getControllerMAC()))) {
            if (this.connectionManager.getLocalityStatus(p.getNode()) == ConnectionLocality.LOCAL) {
                if (log.isTraceEnabled()) {
                    log.trace("Received local ARP req. for default gateway. Replying with controller MAC: {}", (Object)HexEncode.bytesToHexString((byte[])this.getControllerMAC()));
                }
                this.sendARPReply(p, this.getControllerMAC(), targetIP, pkt.getSenderHardwareAddress(), sourceIP);
            } else {
                log.trace("Received non-local ARP req. for default gateway. Raising reply event");
                this.arpRequestReplyEvent.put(new ARPReply(p, targetIP, this.getControllerMAC(), sourceIP, pkt.getSenderHardwareAddress()), false);
            }
            return;
        }
        IHostId id = HostIdFactory.create((InetAddress)targetIP, null);
        HostNodeConnector host = this.hostTracker.hostQuery(id);
        if (host == null) {
            if (requestor != null) {
                Set requestorSet = (Set)this.arpRequestors.get(targetIP);
                if (requestorSet == null) {
                    requestorSet = Collections.newSetFromMap(new ConcurrentHashMap());
                    this.arpRequestors.put(targetIP, requestorSet);
                }
                requestorSet.add(requestor);
                this.countDownTimers.put(targetIP, (short)2);
            }
            log.trace("Sending a bcast ARP request for {}", (Object)targetIP);
            this.arpRequestReplyEvent.put(new ARPRequest(targetIP, subnet), false);
        } else if (NetUtils.isBroadcastMACAddr((byte[])targetMAC) || Arrays.equals(host.getDataLayerAddressBytes(), targetMAC)) {
            log.trace("Received ARP req. for known host {}, sending reply...", (Object)targetIP);
            if (this.connectionManager.getLocalityStatus(p.getNode()) == ConnectionLocality.LOCAL) {
                this.sendARPReply(p, host.getDataLayerAddressBytes(), host.getNetworkAddress(), pkt.getSenderHardwareAddress(), sourceIP);
            } else {
                this.arpRequestReplyEvent.put(new ARPReply(p, host.getNetworkAddress(), host.getDataLayerAddressBytes(), sourceIP, pkt.getSenderHardwareAddress()), false);
            }
        }
    }

    protected void sendBcastARPRequest(InetAddress targetIP, Subnet subnet) {
        HashSet nodeConnectors;
        log.trace("sendBcatARPRequest targetIP:{} subnet:{}", (Object)targetIP, (Object)subnet);
        if (subnet.isFlatLayer2()) {
            nodeConnectors = new HashSet();
            for (Node n : this.switchManager.getNodes()) {
                nodeConnectors.addAll(this.switchManager.getUpNodeConnectors(n));
            }
        } else {
            nodeConnectors = subnet.getNodeConnectors();
        }
        byte[] targetHardwareAddress = new byte[]{0, 0, 0, 0, 0, 0};
        for (NodeConnector p : nodeConnectors) {
            if (this.connectionManager.getLocalityStatus(p.getNode()) != ConnectionLocality.LOCAL || this.topologyManager.isInternal(p)) continue;
            log.trace("Sending toward nodeConnector:{}", (Object)p);
            byte[] senderIP = subnet.getNetworkAddress().getAddress();
            byte[] targetIPByte = targetIP.getAddress();
            ARP arp = this.createARP(ARP.REQUEST, this.getControllerMAC(), senderIP, targetHardwareAddress, targetIPByte);
            byte[] destMACAddress = NetUtils.getBroadcastMACAddr();
            Ethernet ethernet = this.createEthernet(this.getControllerMAC(), destMACAddress, arp);
            RawPacket destPkt = this.dataPacketService.encodeDataPacket((Packet)ethernet);
            destPkt.setOutgoingNodeConnector(p);
            this.dataPacketService.transmitDataPacket(destPkt);
        }
    }

    protected void sendUcastARPRequest(HostNodeConnector host, Subnet subnet) {
        log.trace("sendUcastARPRequest host:{} subnet:{}", (Object)host, (Object)subnet);
        NodeConnector outPort = host.getnodeConnector();
        if (outPort == null) {
            log.error("Failed sending UcastARP because cannot extract output port from Host: {}", (Object)host);
            return;
        }
        byte[] senderIP = subnet.getNetworkAddress().getAddress();
        byte[] targetIP = host.getNetworkAddress().getAddress();
        byte[] targetMAC = host.getDataLayerAddressBytes();
        ARP arp = this.createARP(ARP.REQUEST, this.getControllerMAC(), senderIP, targetMAC, targetIP);
        Ethernet ethernet = this.createEthernet(this.getControllerMAC(), targetMAC, arp);
        RawPacket destPkt = this.dataPacketService.encodeDataPacket((Packet)ethernet);
        destPkt.setOutgoingNodeConnector(outPort);
        this.dataPacketService.transmitDataPacket(destPkt);
    }

    public void find(InetAddress networkAddress) {
        log.trace("Received find IP {}", (Object)networkAddress);
        Subnet subnet = null;
        if (this.switchManager != null) {
            subnet = this.switchManager.getSubnetByNetworkAddress(networkAddress);
        }
        if (subnet == null) {
            log.debug("Can't find subnet matching IP {}", (Object)networkAddress);
            return;
        }
        this.arpRequestReplyEvent.put(new ARPRequest(networkAddress, subnet), false);
    }

    public void probe(HostNodeConnector host) {
        log.trace("Received probe host {}", (Object)host);
        Subnet subnet = null;
        if (this.switchManager != null) {
            subnet = this.switchManager.getSubnetByNetworkAddress(host.getNetworkAddress());
        }
        if (subnet == null) {
            log.debug("can't find subnet matching {}", (Object)host.getNetworkAddress());
            return;
        }
        if (this.connectionManager.getLocalityStatus(host.getnodeconnectorNode()) == ConnectionLocality.LOCAL) {
            log.trace("Send a ucast ARP req. to: {}", (Object)host);
            this.sendUcastARPRequest(host, subnet);
        } else {
            log.trace("Raise a ucast ARP req. event to: {}", (Object)host);
            this.arpRequestReplyEvent.put(new ARPRequest(host, subnet), false);
        }
    }

    protected void handlePuntedIPPacket(IPv4 pkt, NodeConnector p) {
        InetAddress dIP = NetUtils.getInetAddress((int)pkt.getDestinationAddress());
        if (dIP == null) {
            return;
        }
        Subnet subnet = null;
        if (this.switchManager != null) {
            subnet = this.switchManager.getSubnetByNetworkAddress(dIP);
        }
        if (subnet == null) {
            log.debug("Can't find subnet matching {}, drop packet", (Object)dIP);
            return;
        }
        if (subnet.getNetworkAddress().equals(dIP)) {
            log.trace("Ignore IP packet destined to default gw");
            return;
        }
        IHostId id = HostIdFactory.create((InetAddress)dIP, null);
        HostNodeConnector host = this.hostTracker.hostFind(id);
        if (host == null) {
            log.trace("Punted IP pkt to {}, sending bcast ARP event...", (Object)dIP);
            this.arpRequestReplyEvent.put(new ARPRequest(dIP, subnet), false);
        } else if (this.routing == null || this.routing.getRoute(p.getNode(), host.getnodeconnectorNode()) != null) {
            log.trace("forwarding punted IP pkt to {} received at {}", (Object)dIP, (Object)p);
            NodeConnector nc = host.getnodeConnector();
            RawPacket rp = this.dataPacketService.encodeDataPacket(pkt.getParent());
            rp.setOutgoingNodeConnector(nc);
            this.dataPacketService.transmitDataPacket(rp);
        } else {
            log.trace("ignoring punted IP pkt to {} because there is no route from {}", (Object)dIP, (Object)p);
        }
    }

    public byte[] getControllerMAC() {
        if (this.switchManager == null) {
            return null;
        }
        return this.switchManager.getControllerMAC();
    }

    void init() {
        this.arpRequestors = new ConcurrentHashMap<InetAddress, Set<HostNodeConnector>>();
        this.countDownTimers = new ConcurrentHashMap<InetAddress, Short>();
        this.cacheEventHandler = new Thread((Runnable)new ARPCacheEventHandler(), "ARPCacheEventHandler Thread");
        this.allocateCaches();
        this.retrieveCaches();
    }

    private void retrieveCaches() {
        if (this.clusterContainerService == null) {
            log.error("Cluster service unavailable, can't retieve ARPHandler caches!");
            return;
        }
        ConcurrentMap map = this.clusterContainerService.getCache(ARP_EVENT_CACHE_NAME);
        if (map != null) {
            this.arpRequestReplyEvent = map;
        } else {
            log.error("Cache allocation failed for {}", (Object)ARP_EVENT_CACHE_NAME);
        }
    }

    private void allocateCaches() {
        if (this.clusterContainerService == null) {
            this.nonClusterObjectCreate();
            log.error("Clustering service unavailable. Allocated non-cluster caches for ARPHandler.");
            return;
        }
        try {
            this.clusterContainerService.createCache(ARP_EVENT_CACHE_NAME, EnumSet.of(IClusterServices.cacheMode.TRANSACTIONAL));
        }
        catch (CacheConfigException e) {
            log.error("ARPHandler cache configuration invalid!");
        }
        catch (CacheExistException e) {
            log.debug("ARPHandler cache exists, skipped allocation.");
        }
    }

    private void nonClusterObjectCreate() {
        this.arpRequestReplyEvent = new ConcurrentHashMap<ARPEvent, Boolean>();
    }

    void destroy() {
        this.cacheEventHandler.interrupt();
    }

    void start() {
        this.stopping = false;
        this.startPeriodicTimer();
        this.cacheEventHandler.start();
    }

    void stop() {
    }

    void stopping() {
        this.stopping = true;
        this.cancelPeriodicTimer();
    }

    void setSwitchManager(ISwitchManager s) {
        log.debug("SwitchManager service set.");
        this.switchManager = s;
    }

    void unsetSwitchManager(ISwitchManager s) {
        if (this.switchManager == s) {
            log.debug("SwitchManager service UNset.");
            this.switchManager = null;
        }
    }

    public PacketResult receiveDataPacket(RawPacket inPkt) {
        if (inPkt == null) {
            return PacketResult.IGNORED;
        }
        log.trace("Received a frame of size: {}", (Object)inPkt.getPacketData().length);
        Packet formattedPak = this.dataPacketService.decodeDataPacket(inPkt);
        if (formattedPak instanceof Ethernet) {
            Packet nextPak = formattedPak.getPayload();
            if (nextPak instanceof IPv4) {
                log.trace("Handle IP packet: {}", (Object)formattedPak);
                this.handlePuntedIPPacket((IPv4)nextPak, inPkt.getIncomingNodeConnector());
            } else if (nextPak instanceof ARP) {
                log.trace("Handle ARP packet: {}", (Object)formattedPak);
                this.handleARPPacket((Ethernet)formattedPak, (ARP)nextPak, inPkt.getIncomingNodeConnector());
            }
        }
        return PacketResult.IGNORED;
    }

    private ARP createARP(short opCode, byte[] senderMacAddress, byte[] senderIP, byte[] targetMacAddress, byte[] targetIP) {
        ARP arp = new ARP();
        arp.setHardwareType(ARP.HW_TYPE_ETHERNET);
        arp.setProtocolType(EtherTypes.IPv4.shortValue());
        arp.setHardwareAddressLength((byte)6);
        arp.setProtocolAddressLength((byte)4);
        arp.setOpCode(opCode);
        arp.setSenderHardwareAddress(senderMacAddress);
        arp.setSenderProtocolAddress(senderIP);
        arp.setTargetHardwareAddress(targetMacAddress);
        arp.setTargetProtocolAddress(targetIP);
        return arp;
    }

    private Ethernet createEthernet(byte[] sourceMAC, byte[] targetMAC, ARP arp) {
        Ethernet ethernet = new Ethernet();
        ethernet.setSourceMACAddress(sourceMAC);
        ethernet.setDestinationMACAddress(targetMAC);
        ethernet.setEtherType(EtherTypes.ARP.shortValue());
        ethernet.setPayload((Packet)arp);
        return ethernet;
    }

    private void startPeriodicTimer() {
        this.periodicTimer = new Timer("ArpHandler Periodic Timer");
        this.periodicTimer.scheduleAtFixedRate(new TimerTask(){

            @Override
            public void run() {
                Set targetIPs = ArpHandler.this.countDownTimers.keySet();
                HashSet<InetAddress> expiredTargets = new HashSet<InetAddress>();
                for (InetAddress t : targetIPs) {
                    short tick = (Short)ArpHandler.this.countDownTimers.get(t);
                    if ((tick = (short)(tick - 1)) <= 0) {
                        expiredTargets.add(t);
                        continue;
                    }
                    ArpHandler.this.countDownTimers.replace(t, tick);
                }
                for (InetAddress tIP : expiredTargets) {
                    ArpHandler.this.countDownTimers.remove(tIP);
                    ArpHandler.this.arpRequestors.remove(tIP);
                    log.debug("ARP reply was not received from {}", (Object)tIP);
                }
                try {
                    if (ArpHandler.this.clusterContainerService.amICoordinator() && !ArpHandler.this.arpRequestReplyEvent.isEmpty()) {
                        ArpHandler.this.arpRequestReplyEvent.clear();
                    }
                }
                catch (Exception e) {
                    log.warn("ARPHandler: A cluster member failed to clear event cache.");
                }
            }
        }, 0L, 1000L);
    }

    private void cancelPeriodicTimer() {
        if (this.periodicTimer != null) {
            this.periodicTimer.cancel();
        }
    }

    private void generateAndSendReply(InetAddress sourceIP, byte[] sourceMAC) {
        Set hosts;
        if (log.isTraceEnabled()) {
            log.trace("generateAndSendReply called with params sourceIP:{} sourceMAC:{}", (Object)sourceIP, (Object)HexEncode.bytesToHexString((byte[])sourceMAC));
        }
        if ((hosts = (Set)this.arpRequestors.remove(sourceIP)) == null || hosts.isEmpty()) {
            log.trace("Bailing out no requestors Hosts");
            return;
        }
        this.countDownTimers.remove(sourceIP);
        for (HostNodeConnector host : hosts) {
            if (log.isTraceEnabled()) {
                log.trace("Sending ARP Reply with src {}/{}, target {}/{}", new Object[]{HexEncode.bytesToHexString((byte[])sourceMAC), sourceIP, HexEncode.bytesToHexString((byte[])host.getDataLayerAddressBytes()), host.getNetworkAddress()});
            }
            if (this.connectionManager.getLocalityStatus(host.getnodeconnectorNode()) == ConnectionLocality.LOCAL) {
                this.sendARPReply(host.getnodeConnector(), sourceMAC, sourceIP, host.getDataLayerAddressBytes(), host.getNetworkAddress());
                continue;
            }
            this.arpRequestReplyEvent.put(new ARPReply(host.getnodeConnector(), sourceIP, sourceMAC, host.getNetworkAddress(), host.getDataLayerAddressBytes()), false);
        }
    }

    public void entryUpdated(ARPEvent key, Boolean new_value, String cacheName, boolean originLocal) {
        log.trace("Got and entryUpdated for cacheName {} key {} isNew {}", new Object[]{cacheName, key, new_value});
        this.enqueueARPCacheEvent(key, new_value);
    }

    public void entryCreated(ARPEvent key, String cacheName, boolean originLocal) {
    }

    public void entryDeleted(ARPEvent key, String cacheName, boolean originLocal) {
    }

    private void enqueueARPCacheEvent(ARPEvent event, boolean new_value) {
        try {
            ARPCacheEvent cacheEvent = new ARPCacheEvent(event, new_value);
            if (!this.ARPCacheEvents.contains(cacheEvent)) {
                this.ARPCacheEvents.add(cacheEvent);
                log.trace("Enqueued {}", (Object)event);
            }
        }
        catch (Exception e) {
            log.debug("enqueueARPCacheEvent caught Interrupt Exception for event {}", (Object)event);
        }
    }

    private class ARPCacheEventHandler
    implements Runnable {
        private ARPCacheEventHandler() {
        }

        @Override
        public void run() {
            while (!ArpHandler.this.stopping) {
                try {
                    ARPCacheEvent ev = (ARPCacheEvent)ArpHandler.this.ARPCacheEvents.take();
                    ARPEvent event = ev.getEvent();
                    if (event instanceof ARPRequest) {
                        ARPRequest req = (ARPRequest)event;
                        if (req.getHost() == null) {
                            log.trace("Trigger and ARP Broadcast Request upon receipt of {}", (Object)req);
                            ArpHandler.this.sendBcastARPRequest(req.getTargetIP(), req.getSubnet());
                            continue;
                        }
                        if (ArpHandler.this.connectionManager.getLocalityStatus(req.getHost().getnodeconnectorNode()) != ConnectionLocality.LOCAL) continue;
                        log.trace("ARPCacheEventHandler - sendUcatARPRequest upon receipt of {}", (Object)req);
                        ArpHandler.this.sendUcastARPRequest(req.getHost(), req.getSubnet());
                        continue;
                    }
                    if (!(event instanceof ARPReply)) continue;
                    ARPReply rep = (ARPReply)event;
                    if (ev.isNewReply()) {
                        log.trace("Trigger a generateAndSendReply in response to {}", (Object)rep);
                        ArpHandler.this.generateAndSendReply(rep.getTargetIP(), rep.getTargetMac());
                        continue;
                    }
                    if (ArpHandler.this.connectionManager.getLocalityStatus(rep.getPort().getNode()) != ConnectionLocality.LOCAL) continue;
                    log.trace("ARPCacheEventHandler - sendUcatARPReply locally in response to {}", (Object)rep);
                    ArpHandler.this.sendARPReply(rep.getPort(), rep.getSourceMac(), rep.getSourceIP(), rep.getTargetMac(), rep.getTargetIP());
                }
                catch (InterruptedException e) {
                    ArpHandler.this.ARPCacheEvents.clear();
                    return;
                }
            }
        }
    }
}

