/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.controller.netconf.persist.impl;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import io.netty.channel.EventLoopGroup;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetSocketAddress;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import javax.annotation.concurrent.Immutable;
import org.opendaylight.controller.config.api.ConflictingVersionException;
import org.opendaylight.controller.config.persist.api.ConfigSnapshotHolder;
import org.opendaylight.controller.netconf.api.NetconfMessage;
import org.opendaylight.controller.netconf.client.NetconfClient;
import org.opendaylight.controller.netconf.client.NetconfClientDispatcher;
import org.opendaylight.controller.netconf.persist.impl.ConfigPersisterNotificationHandler;
import org.opendaylight.controller.netconf.persist.impl.Util;
import org.opendaylight.controller.netconf.util.NetconfUtil;
import org.opendaylight.controller.netconf.util.messages.NetconfMessageAdditionalHeader;
import org.opendaylight.controller.netconf.util.xml.XmlElement;
import org.opendaylight.controller.netconf.util.xml.XmlUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

@Immutable
public class ConfigPusher {
    private static final Logger logger = LoggerFactory.getLogger(ConfigPersisterNotificationHandler.class);
    private static final int NETCONF_SEND_ATTEMPT_MS_DELAY = 1000;
    private static final int NETCONF_SEND_ATTEMPTS = 20;
    private final InetSocketAddress address;
    private final EventLoopGroup nettyThreadgroup;
    public static final long DEFAULT_TIMEOUT = TimeUnit.MINUTES.toNanos(2L);
    private final int delayMillis = 5000;
    private final long timeoutNanos;

    public ConfigPusher(InetSocketAddress address, EventLoopGroup nettyThreadgroup) {
        this(address, nettyThreadgroup, DEFAULT_TIMEOUT);
    }

    @Deprecated
    public ConfigPusher(InetSocketAddress address, long timeoutMillis, EventLoopGroup nettyThreadgroup) {
        this(address, nettyThreadgroup, TimeUnit.MILLISECONDS.toNanos(timeoutMillis));
    }

    public ConfigPusher(InetSocketAddress address, EventLoopGroup nettyThreadgroup, long timeoutNanos) {
        this.address = address;
        this.nettyThreadgroup = nettyThreadgroup;
        this.timeoutNanos = timeoutNanos;
    }

    public synchronized NetconfClient init(List<ConfigSnapshotHolder> configs) throws InterruptedException {
        logger.debug("Last config snapshots to be pushed to netconf: {}", configs);
        return this.pushAllConfigs(configs);
    }

    private synchronized NetconfClient pushAllConfigs(List<ConfigSnapshotHolder> configs) throws InterruptedException {
        NetconfClient netconfClient = this.makeNetconfConnection(Collections.emptySet(), (Optional<NetconfClient>)Optional.absent());
        for (ConfigSnapshotHolder configSnapshotHolder : configs) {
            netconfClient = this.pushSnapshotWithRetries(configSnapshotHolder, (Optional<NetconfClient>)Optional.of((Object)netconfClient));
            logger.debug("Config snapshot pushed successfully: {}", (Object)configSnapshotHolder);
        }
        logger.debug("All configuration snapshots have been pushed successfully.");
        return netconfClient;
    }

    private synchronized NetconfClient pushSnapshotWithRetries(ConfigSnapshotHolder configSnapshotHolder, Optional<NetconfClient> oldClientForPossibleReuse) throws InterruptedException {
        Throwable lastException = null;
        int maxAttempts = 30;
        for (int i = 0; i < maxAttempts; ++i) {
            NetconfClient netconfClient = this.makeNetconfConnection(configSnapshotHolder.getCapabilities(), oldClientForPossibleReuse);
            logger.trace("Pushing following xml to netconf {}", (Object)configSnapshotHolder);
            try {
                this.pushLastConfig(configSnapshotHolder, netconfClient);
                return netconfClient;
            }
            catch (IOException | ConflictingVersionException e) {
                Util.closeClientAndDispatcher(netconfClient);
                lastException = e;
                Thread.sleep(1000L);
                continue;
            }
            catch (SAXException e) {
                throw new IllegalStateException("Unable to load last config", e);
            }
        }
        throw new IllegalStateException("Failed to push configuration, maximum attempt count has been reached: " + maxAttempts, lastException);
    }

    private synchronized NetconfClient makeNetconfConnection(Set<String> expectedCaps, Optional<NetconfClient> maybeOldClient) throws InterruptedException {
        if (maybeOldClient.isPresent()) {
            NetconfClient oldClient = (NetconfClient)maybeOldClient.get();
            Util.closeClientAndDispatcher(oldClient);
        }
        long pollingStart = System.nanoTime();
        long deadline = pollingStart + this.timeoutNanos;
        int attempt = 0;
        String additionalHeader = NetconfMessageAdditionalHeader.toString((String)"unknown", (String)this.address.getAddress().getHostAddress(), (String)Integer.toString(this.address.getPort()), (String)"tcp", (Optional)Optional.of((Object)"persister"));
        Set latestCapabilities = new HashSet();
        while (System.nanoTime() < deadline) {
            NetconfClient netconfClient;
            ++attempt;
            NetconfClientDispatcher netconfClientDispatcher = new NetconfClientDispatcher(this.nettyThreadgroup, this.nettyThreadgroup, additionalHeader);
            try {
                netconfClient = new NetconfClient(this.toString(), this.address, 5000, netconfClientDispatcher);
            }
            catch (IllegalStateException e) {
                logger.debug("Netconf {} was not initialized or is not stable, attempt {}", new Object[]{this.address, attempt, e});
                netconfClientDispatcher.close();
                Thread.sleep(5000L);
                continue;
            }
            latestCapabilities = netconfClient.getCapabilities();
            if (Util.isSubset(netconfClient, expectedCaps)) {
                logger.debug("Hello from netconf stable with {} capabilities", latestCapabilities);
                logger.trace("Session id received from netconf server: {}", (Object)netconfClient.getClientSession());
                return netconfClient;
            }
            logger.debug("Polling hello from netconf, attempt {}, capabilities {}", (Object)attempt, latestCapabilities);
            Util.closeClientAndDispatcher(netconfClient);
            Thread.sleep(5000L);
        }
        HashSet<String> allNotFound = new HashSet<String>(expectedCaps);
        allNotFound.removeAll(latestCapabilities);
        logger.error("Netconf server did not provide required capabilities. Expected but not found: {}, all expected {}, current {}", new Object[]{allNotFound, expectedCaps, latestCapabilities});
        throw new RuntimeException("Netconf server did not provide required capabilities. Expected but not found:" + allNotFound);
    }

    private synchronized void pushLastConfig(ConfigSnapshotHolder configSnapshotHolder, NetconfClient netconfClient) throws ConflictingVersionException, IOException, SAXException {
        Element xmlToBePersisted = XmlUtil.readXmlToElement((String)configSnapshotHolder.getConfigSnapshot());
        logger.trace("Pushing last configuration to netconf: {}", (Object)configSnapshotHolder);
        StringBuilder response = new StringBuilder("editConfig response = {");
        NetconfMessage message = ConfigPusher.createEditConfigMessage(xmlToBePersisted, "/netconfOp/editConfig.xml");
        NetconfMessage responseMessage = ConfigPusher.getResponse(message, netconfClient);
        NetconfUtil.checkIsMessageOk((NetconfMessage)responseMessage);
        response.append(XmlUtil.toString((Document)responseMessage.getDocument()));
        response.append("}");
        responseMessage = ConfigPusher.getResponse(ConfigPusher.getNetconfMessageFromResource("/netconfOp/commit.xml"), netconfClient);
        NetconfUtil.checkIsMessageOk((NetconfMessage)responseMessage);
        response.append("commit response = {");
        response.append(XmlUtil.toString((Document)responseMessage.getDocument()));
        response.append("}");
        logger.trace("Last configuration loaded successfully");
        logger.trace("Detailed message {}", (Object)response);
    }

    private static NetconfMessage getResponse(NetconfMessage request, NetconfClient netconfClient) throws IOException {
        try {
            return netconfClient.sendMessage(request, 20, 1000);
        }
        catch (RuntimeException e) {
            logger.debug("Error while executing netconf transaction {} to {}", new Object[]{request, netconfClient, e});
            throw new IOException("Failed to execute netconf transaction", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static NetconfMessage createEditConfigMessage(Element dataElement, String editConfigResourcename) throws IOException, SAXException {
        try (InputStream stream = ConfigPersisterNotificationHandler.class.getResourceAsStream(editConfigResourcename);){
            Preconditions.checkNotNull((Object)stream, (Object)("Unable to load resource " + editConfigResourcename));
            Document doc = XmlUtil.readXmlToDocument((InputStream)stream);
            doc.getDocumentElement();
            XmlElement editConfigElement = XmlElement.fromDomDocument((Document)doc).getOnlyChildElement();
            XmlElement configWrapper = editConfigElement.getOnlyChildElement("config");
            editConfigElement.getDomElement().removeChild(configWrapper.getDomElement());
            for (XmlElement el : XmlElement.fromDomElement((Element)dataElement).getChildElements()) {
                configWrapper.appendChild((Element)doc.importNode(el.getDomElement(), true));
            }
            editConfigElement.appendChild(configWrapper.getDomElement());
            NetconfMessage netconfMessage = new NetconfMessage(doc);
            return netconfMessage;
        }
        catch (IOException | SAXException e) {
            logger.debug("Failed to create edit-config message for resource {}", (Object)editConfigResourcename, (Object)e);
            throw e;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static NetconfMessage getNetconfMessageFromResource(String resource) throws IOException, SAXException {
        try (InputStream stream = ConfigPusher.class.getResourceAsStream(resource);){
            Preconditions.checkNotNull((Object)stream, (Object)("Unable to load resource " + resource));
            NetconfMessage netconfMessage = new NetconfMessage(XmlUtil.readXmlToDocument((InputStream)stream));
            return netconfMessage;
        }
        catch (IOException | SAXException e) {
            logger.debug("Failed to parse netconf message for resource {}", (Object)resource, (Object)e);
            throw e;
        }
    }
}

